#!/bin/sh
#
# Make USB key to save/restore HW configuration on a ProLiant DL 680 G7
#
#set -x

sync
DRIVE=$1
[ -z $DRIVE ] && echo "Usage: $0 device." && exit -1
[ ! -b $DRIVE ] && echo "$1 is not a block device. Exiting." && exit -1
KPARTX=`which kpartx`
[ $? -ne 0 ] && echo "kpartx is missing. exiting" && exit -1
SYSLINUX=`which syslinux`
[ $? -ne 0 ] && echo "syslinux is missing. exiting" && exit -1


[[ $DRIVE == /dev/sda* ]] && echo "For security, the script does not work on /dev/sda* devices" && exit -1

[[ `cat /proc/mounts | grep $DRIVE` != "" ]] && echo "$DRIVE is mounted. Exit now." && exit -1

# Wipe the partition table.
dd if=/dev/zero of=$DRIVE bs=4096 count=1 2>&1 > /dev/null
[ $? -ne 0 ] && echo "WARNING: Unable to erase MBR on $DRIVE."

# Create partition
echo "Creating the partition on $DRIVE"
sfdisk -q -D $DRIVE << EOF
0,,C,*
EOF
[ $? -ne 0 ] && echo "Unable to partition the key $DRIVE. Exiting" && exit -1
sfdisk -q -R $DRIVE

if [[ $DRIVE == /dev/loop* ]]; then
	$KPARTX -av $DRIVE
	PART="/dev/mapper/${DRIVE#/dev/}p1"
else
	PART=${DRIVE}1
fi

echo "Creating the master boot record on $DRIVE"
dd if=./mbr.bin of=$DRIVE > 2>&1 /dev/null
[ $? -ne 0 ] && echo "Unable to copy MBR on $DRIVE. Please check the key. Exiting" && exit -1

# FAT32 filesystem labeled 'LiveUSB'
sleep 5
echo "Creating FAT filesystem on $PART"
mkfs.vfat -n LiveUSB $PART > /dev/null
[ $? -ne 0 ] && echo "Unable to format the partition $PART as VFAT. Exiting" && exit -1

# The SQUASHFS containing the EXT3 filesystem with the
# minimum OS and the SSTK tools
SQUASHFS=$(mktemp -d squashfs.XXXX)
mkdir $SQUASHFS/LiveOS
dd if=ext3fs.img of=$SQUASHFS/LiveOS/ext3fs.img 2>&1 > /dev/null
[ $? -ne 0 ] && echo "Unable to copy the squashfs FS. Exiting" && exit -1

TMPDIR=`mktemp -d mkusb.XXXX`
if [ ! -d "$TMPDIR" ]; then
	echo "$TMPDIR is not a directory. Exiting"
	exit -1
else
	if [[ "$TMPDIR" == "/" ]]; then
		echo "$TMPDIR is /. Exiting"
		exit -1
	fi
fi

ret=0
mount $PART $TMPDIR
if [ $? -ne 0 ] ; then
	echo "Unable to mount $PART under $TMPDIR. Exiting" 
	ret=1
else
	mkdir $TMPDIR/syslinux $TMPDIR/LiveOS $TMPDIR/data_files $TMPDIR/fw_files $TMPDIR/log
	cat > $TMPDIR/fw_files/README << EOF
ProLIant Linux entry point:
http://www.hp.com/go/proliantlinux

Firmware update for the DL 380 G7 are available at
http://h20000.www2.hp.com/bizsupport/TechSupport/SoftwareIndex.jsp?lang=en&cc=us&prodNameId=4091432&prodTypeId=15351&prodSeriesId=4091412&swLang=8&taskId=135&swEnvOID=4103

Need to download the various .scexe files and place them in that directory so the server is being updated when choosing the fw option in boot menu.

Latest FW available at time of USB Key built:
BIOS P67 2011.05.05 (CP015473.scexe)
iLO3 1.20 2011-04-05 (CP014002.scexe)
Smart Array 5.06 2011-08-16 (CP015807.scexe)
Embedded Broadcom NIC 2.3.0 2011-02-24 (CP014348.scexe)
NC364T NIC - None available
NC365T NIC - None available
EOF
	echo "Copying system files..."
	$SYSLINUX -i -d syslinux $PART
	if [ $? -ne 0 ]; then
		echo "Unable to syslinux $PART. Exiting"
		ret=1
	else
		cp vmlinuz initramfs.img syslinux.cfg $TMPDIR/syslinux
		cp fw_files/* $TMPDIR/fw_files
		cp data_files/* $TMPDIR/data_files
		mksquashfs $SQUASHFS $TMPDIR/LiveOS/squashfs.img
		if [ $? -ne 0 ]; then
			echo "Unable to recreate the squash FS $SQUASHFS. Exiting"
			ret=1
		fi
	fi
fi
rm -rf $SQUASHFS
if [ $ret -eq 1 ]; then
	umount $TMPDIR
	exit $ret
fi

echo "Done"

echo "Unmounting file system. It may take some time to flush the buffers..."
umount $TMPDIR

[[ $PART == "/dev/mapper/loop*" ]] && $KPARTX -dv $DRIVE

rm -rf $TMPDIR
