#!/usr/bin/perl -w
#
# ProjectBuilder Conf module
# Conf files subroutines brought by the the Project-Builder project
# which can be easily used by wahtever perl project
#
# $Id$
#

package ProjectBuilder::Conf;

use strict;
use Data::Dumper;
use ProjectBuilder::Base;

# Inherit from the "Exporter" module which handles exporting functions.
 
use Exporter;
 
# Export, by default, all the functions into the namespace of
# any code which uses this module.
 
our @ISA = qw(Exporter);
our @EXPORT = qw(pb_conf_init pb_conf_read pb_conf_read_if pb_conf_get pb_conf_get_if);

=pod

=head1 NAME

ProjectBuilder::Conf, part of the project-builder.org - module dealing with configuration files

=head1 DESCRIPTION

This modules provides functions dealing with configuration files.

=head1 SYNOPSIS

  use ProjectBuilder::Conf;

  #
  # Read hash codes of values from a configuration file and return table of pointers
  #
  my ($k1, $k2) = pb_conf_read_if("$ENV{'HOME'}/.pbrc","key1","key2");
  my ($k) = pb_conf_read("$ENV{'HOME'}/.pbrc","key");

=head1 USAGE

=over 4

=item B<pb_conf_init>

This function initializes the configuration files analysis.
Pass the configuration files to consider in order of importance (the most important first).

=cut

sub pb_conf_init {

my @conffiles = @_;
}

=item B<pb_conf_read_if>

This function returns a table of pointers on hashes
corresponding to the keys in a configuration file passed in parameter.
If that file doesn't exist, it returns undef.

The format of the configuration file is as follows:

key tag = value1,value2,...

Supposing the file is called "$ENV{'HOME'}/.pbrc", containing the following:

  $ cat $HOME/.pbrc
  pbver pb = 3
  pbver default = 1
  pblist pb = 12,25

calling it like this:

  my ($k1, $k2) = pb_conf_read_if("$ENV{'HOME'}/.pbrc","pbver","pblist");

will allow to get the mapping:

  $k1->{'pb'}  contains 3
  $ka->{'default'} contains 1
  $k2->{'pb'} contains 12,25

Valid chars for keys and tags are letters, numbers, '-' and '_'.

=cut

sub pb_conf_read_if {

my $conffile = shift;
my @param = @_;

open(CONF,$conffile) || return((undef));
close(CONF);
return(pb_conf_read($conffile,@param));
}

=item B<pb_conf_read>

This function is similar to B<pb_conf_read_if> except that it dies when the file in parameter doesn't exist.

=cut

sub pb_conf_read {

my $conffile = shift;
my @param = @_;
my $trace;
my @ptr;
my %h;

open(CONF,$conffile) || die "Unable to open $conffile";
while(<CONF>) {
	if (/^\s*([A-z0-9-_]+)\s+([[A-z0-9-_]+)\s*=\s*(.+)$/) {
		pb_log(3,"DEBUG: 1:$1 2:$2 3:$3\n");
		$h{$1}{$2}=$3;
	}
}
close(CONF);

for my $param (@param) {
	push @ptr,$h{$param};
}
return(@ptr);
}


# Function which returns a pointer on a table
# corresponding to a set of values queried in the conf file
# and test the returned vaue as they need to exist in that case
sub pb_conf_get {

my @param = @_;
my @return = pb_conf_get_if(@param);

die "No params found for $ENV{'PBPROJ'}" if (not @return);

foreach my $i (0..$#param) {
	die "No $param[$i] defined for $ENV{'PBPROJ'}" if (not defined $return[$i]);
}
return(@return);
}

# Function which returns a pointer on a table
# corresponding to a set of values queried in the conf file
# Those value may be undef if they do not exist
sub pb_conf_get_if {

my @param = @_;

# Everything is returned via ptr1
my @ptr1 = ();
my @ptr2 = ();
@ptr1 = pb_conf_read_if("$ENV{'PBETC'}", @param) if (defined $ENV{'PBETC'});
@ptr2 = pb_conf_read_if("$ENV{'PBROOTDIR'}/$ENV{'PBPROJ'}.pb", @param) if ((defined $ENV{'PBROOTDIR'}) and (defined $ENV{'PBPROJ'}));

my $p1;
my $p2;

pb_log(2,"DEBUG: pb_conf_get param1: ".Dumper(@ptr1)."\n");
pb_log(2,"DEBUG: pb_conf_get param2: ".Dumper(@ptr2)."\n");

foreach my $i (0..$#param) {
	$p1 = $ptr1[$i];
	$p2 = $ptr2[$i];
	# Always try to take the param from the home dir conf file in priority
	# in order to mask what could be defined under the CMS to allow for overloading
	if (not defined $p2) {
		# No ref in CMS project conf file so use the home dir one.
		$p1->{$ENV{'PBPROJ'}} = $p1->{'default'} if ((not defined $p1->{$ENV{'PBPROJ'}}) && (defined $p1->{'default'}));
	} else {
		# Ref found in CMS project conf file
		if (not defined $p1) {
			# No ref in home dir project conf file so use the CMS one.
			$p2->{$ENV{'PBPROJ'}} = $p2->{'default'} if ((not defined $p2->{$ENV{'PBPROJ'}}) && (defined $p2->{'default'}));
			$p1 = $p2;
		} else {
			# Both are defined - handling the overloading
			if (not defined $p1->{'default'}) {
				if (defined $p2->{'default'}) {
					$p1->{'default'} = $p2->{'default'};
				}
			}

			if (not defined $p1->{$ENV{'PBPROJ'}}) {
				if (defined $p2->{$ENV{'PBPROJ'}}) {
					$p1->{$ENV{'PBPROJ'}} = $p2->{$ENV{'PBPROJ'}} if (defined $p2->{$ENV{'PBPROJ'}});
				} else {
					$p1->{$ENV{'PBPROJ'}} = $p1->{'default'} if (defined $p1->{'default'});
				}
			}
			# Now copy back into p1 all p2 content which doesn't exist in p1
			# p1 content (local) always has priority over p2 (project)
			foreach my $k (keys %$p2) {
				$p1->{$k} = $p2->{$k} if (not defined $p1->{$k});
			}
		}
	}
	$ptr1[$i] = $p1;
}
pb_log(2,"DEBUG: pb_conf_get param ptr1: ".Dumper(@ptr1)."\n");
return(@ptr1);
}


=back 

=head1 WEB SITES

The main Web site of the project is available at L<http://www.project-builder.org/>. Bug reports should be filled using the trac instance of the project at L<http://trac.project-builder.org/>.

=head1 USER MAILING LIST

None exists for the moment.

=head1 AUTHORS

The Project-Builder.org team L<http://trac.project-builder.org/> lead by Bruno Cornec L<mailto:bruno@project-builder.org>.

=head1 COPYRIGHT

Project-Builder.org is distributed under the GPL v2.0 license
described in the file C<COPYING> included with the distribution.

=cut


1;
