#!/usr/bin/perl -w
#
# Base subroutines for the Project-Builder project
#
# $Id$
#

use strict;
use lib qw (lib);
use File::Basename;
use File::Path;
use File::Temp qw /tempdir/;
use AppConfig qw(ARGCOUNT_HASH);
use Data::Dumper;

$ENV{'PBETC'} = "$ENV{'HOME'}/.pbrc";

sub pb_env_init {

my $proj=shift;
my $ver;
my $tag;

#
# Check project name
# Could be with env var PBPROJ
# or option -p
# if not define take the first in conf file
#
if ((defined $ENV{'PBPROJ'}) &&
	(not (defined $proj))) {
	$proj = $ENV{'PBPROJ'};
}

#
# We get the pbrc file for that project 
# and use its content
#
my ($pbrc) = pb_conf_read("$ENV{'PBETC'}","pbrc");
print "DEBUG pbrc: ".Dumper($pbrc)."\n" if ($debug >= 1);

%pbrc = %$pbrc;
if (not defined $proj) {
	# Take the first as the default project
	$proj = (keys %pbrc)[0];
	print $LOG "Using $proj as default project as none has been specified\n" if (($debug >= 0) and (defined $proj));
}
die "No project defined - use env var PBPROJ or -p proj" if (not (defined $proj));

#
# Set delivery directory
#
my $topdir=dirname($pbrc{$proj});
chdir $topdir || die "Unable to change directory to $topdir";
$ENV{'PBDESTDIR'}=$topdir."/delivery";

#
# Use project configuration file if needed
#
if (not defined $ENV{'PBROOT'}) {
	if (-f $pbrc{$proj}) {
		my ($pbroot) = pb_conf_read($pbrc{$proj},"pbroot");
		my %pbroot = %$pbroot;
		# All lines should point to the same pbroot so take the first
		$ENV{'PBROOT'} = (values %$pbroot)[0] if (defined $pbroot); 
		print $LOG "Using $ENV{'PBROOT'} as default pbroot from $pbrc{$proj}\n" if (($debug >= 0) and (defined $ENV{'PBROOT'}));
	}
	die "No pbroot defined - use env var PBROOT or -r pbroot " if (not defined $ENV{'PBROOT'});
}

#
# Check pb conf compliance
#
$ENV{'PBCONF'} = "$ENV{'PBROOT'}/pbconf";
die "Project $proj not Project-Builder compliant. Please populate $ENV{'PBCONF'}" if ( not -d "$ENV{'PBCONF'}");

my %version = ();

if (-f "$ENV{'PBCONF'}/$proj.pb") {
	# List of pkg to build by default (mandatory)
	# List of additional pkg to build when all is called (optional)
	# Valid version names (optional)
	# List of files to filter (optional)
	my ($defpkgdir, $extpkgdir, $version, $filteredfiles) = pb_conf_read("$ENV{'PBCONF'}/$proj.pb","defpkgdir","extpkgdir","version","filteredfiles");
	print "DEBUG: defpkgdir: ".Dumper($defpkgdir)."\n" if ($debug >= 1);
	print "DEBUG: extpkgdir: ".Dumper($extpkgdir)."\n" if ($debug >= 1);
	print "DEBUG: version: ".Dumper($version)."\n" if ($debug >= 1);
	print "DEBUG: filteredfiles: ".Dumper($filteredfiles)."\n" if ($debug >= 1);
	die "Unable to find defpkgdir in $ENV{'PBCONF'}/$proj.pb" if (not defined $defpkgdir);
	# Global
	%defpkgdir = %$defpkgdir;
	# Global
	%extpkgdir = ();
	%extpkgdir = %$defpkgdir if (defined $defpkgdir);
	%version = ();
	%version = %$version if (defined $version);
	# Global
	%filteredfiles = ();
	%filteredfiles = %$filteredfiles if (defined $filteredfiles);
} else {
	die "Unable to open $ENV{'PBCONF'}/$proj.pb";
}

#
# Set temp directory
#
if (not defined $ENV{'TMPDIR'}) {
	$ENV{'TMPDIR'}="/tmp";
}
$ENV{'PBTMP'} = tempdir( "pb.XXXXXXXXXX", DIR => $ENV{'TMPDIR'}, CLEANUP => 1 );

#
# Get global VERSION
#
open(VER, "$ENV{'PBCONF'}/VERSION") || die "Unable to open $ENV{'PBCONF'}/VERSION: $?";
$ver = <VER>;
chomp($ver);
#print Dumper(%version);
die "Invalid version name $ver in $ENV{'PBROOT'}/VERSION" if ($ver !~ /[0-9.]+/) && (not exists $version{$ver});
$ENV{'PBVER'}=$ver;
close(VER);

#
# Get global TAG
#
open(TAG, "$ENV{'PBCONF'}/TAG") || die "Unable to open $ENV{'PBCONF'}/TAG: $?";
$tag = <TAG>;
chomp($tag);
die "Invalid tag name $tag in $ENV{'PBROOT'}/TAG" if ($tag !~ /[0-9]+/);
$ENV{'PBTAG'}=$tag;
close(TAG);

#
# Removes all directory existing below the delivery dir 
# as they are temp dir only
# Files stay and have to be cleaned up manually
#
if (-d $ENV{'PBDESTDIR'}) {
	opendir(DIR,$ENV{'PBDESTDIR'}) || die "Unable to open directory $ENV{'PBDESTDIR'}: $!";
	foreach my $d (readdir(DIR)) {
		next if ($d =~ /^\./);
		next if (-f "$ENV{'PBDESTDIR'}/$d");
		pb_rm_rf("$ENV{'PBDESTDIR'}/$d") if (-d "$ENV{'PBDESTDIR'}/$d");
	}
	closedir(DIR);
}
if (! -d "$ENV{'PBDESTDIR'}") {
	pb_mkdir_p($ENV{'PBDESTDIR'}) || die "Unable to recursively create $ENV{'PBDESTDIR'}";
}

#
# Set build directory
#
$ENV{'PBBUILDDIR'}=$topdir."/build";
pb_rm_rf($ENV{'PBBUILDDIR'}) if (-d "$ENV{'PBBUILDDIR'}");
pb_mkdir_p($ENV{'PBBUILDDIR'}) || die "Unable to recursively create $ENV{'PBBUILDDIR'}";

umask 0022;
return($proj);
}

# Internal mkdir -p function
sub pb_mkdir_p {
my @dir = @_;
my $ret = mkpath(@dir, 0, 0755);
return($ret);
}

# Internal rm -rf function
sub pb_rm_rf {
my @dir = @_;
my $ret = rmtree(@dir, 0, 0);
return($ret);
}

# Internal system function
sub pb_system {

my $cmd=shift;
my $cmt=shift || $cmd;

print $LOG "$cmt... ";
system("$cmd");
if ($? == -1) {
	print $LOG "failed to execute: $!\n" if ($debug >= 0);
} elsif ($? & 127) {
	printf $LOG "child died with signal %d, %s coredump\n", ($? & 127),  ($? & 128) ? 'with' : 'without' if ($debug >= 0);
} else {
	print $LOG "OK\n" if ($debug >= 0);
}
}

# Function which returns a pointer on a hash
# corresponding to a declaration (arg2) in the main conf file
# and test the returned vaue as they need to exist in that case
sub pb_conf_get {

my @param = @_;

my @ptr = pb_conf_read("$ENV{'PBETC'}", @param);

foreach my $i (0..$#param) {
	die "No $param[$i] defined for $ENV{'PBPROJ'}" if (not defined $ptr[$i]);
	my $p = $ptr[$i];
	$p->{$ENV{'PBPROJ'}} = $p->{'default'} if (not defined $p->{$ENV{'PBPROJ'}});
	die "No $param[$i] defined for $ENV{'PBPROJ'}" if (not defined $p->{$ENV{'PBPROJ'}});
}
print "DEBUG: param: ".Dumper($ptr)."\n" if ($debug >= 1);
return(@ptr);
}

# Function which returns a pointer on a hash
# corresponding to a declaration (arg2) in a conf file (arg1)
sub pb_conf_read {

my $conffile = shift;
my @param = @_;
my $trace;
my @ptr;

if ($debug > 0) {
	$trace = 1;
} else {
	$trace = 0;
}


my $config = AppConfig->new({
							# Auto Create variables mentioned in Conf file
							CREATE => 1,
							DEBUG => $trace,
							GLOBAL => {
								# Each conf item is a hash
								ARGCOUNT => ARGCOUNT_HASH,
							},
						});
$config->file($conffile);
for my $param (@param) {
	push @ptr,$config->get($param);
}
print "DEBUG: params: ".Dumper(@param)." ".Dumper(@ptr)."\n" if ($debug >= 1);
return(@ptr);
}

# Setup environment for CMS system
sub pb_cms_init {

my $proj = shift || undef;
my $ret;

my ($cms) = pb_conf_get("cms");
# This one is optional
my ($cvsroot) = pb_conf_read($ENV{'PBETC'},"cvsroot");

if ($cms->{$proj} eq "svn") {
	$ENV{'PBREVISION'}=`(cd "$ENV{'PBROOT'}" ; svnversion .)`;
	chomp($ENV{'PBREVISION'});
	$ENV{'PBCMSLOG'}="svn log";
	$ENV{'PBCMSLOGFILE'}="svn.log";
	$ENV{'PBCMSEXP'}="svn export";
} elsif ($cms->{$proj} eq "cvs") {
	$ENV{'PBREVISION'}=`(cd "$ENV{'PBROOT'}" ; cvs rannotate  -f . 2>&1 | awk '{print \$1}' | grep -E '^[0-9]' | cut -d. -f2 |sort -nu | tail -1)`;
	chomp($ENV{'PBREVISION'});
	$ENV{'PBCMSLOG'}="cvs log";
	$ENV{'PBCMSLOGFILE'}="cvs.log";
	$ENV{'PBCMSEXP'}="cvs export";
	#
	# Export content if needed
	#
	$ENV{'CVSROOT'} = $cvsroot->{$proj} if (defined $cvsroot->{$proj});
} else {
	die "cms $cms->{$proj} unknown";
}
}

# Get all filters to apply
# They're cumulative from less specific to most specific
# suffix is .pbf

sub pb_get_filters {

my @ffiles;
my ($ffile0, $ffile1, $ffile2, $ffile3);
my $pbpkg = shift || die "No package specified";
my $dtype = shift || die "No dtype specified";
my $dfam = shift || die "No dfam specified";
my $ddir = shift || die "No ddir specified";
my $dver = shift || die "No dver specified";
my $ptr; # returned value pointer on the hash of filters
my %ptr;

if (-d "$ENV{'PBCONF'}/$pbpkg/pbfilter") {
	$ffile0 = "$ENV{'PBCONF'}/$pbpkg/pbfilter/$dtype.pbf" if (-f "$ENV{'PBCONF'}/$pbpkg/pbfilter/$dtype.pbf");
	$ffile1 = "$ENV{'PBCONF'}/$pbpkg/pbfilter/$dfam.pbf" if (-f "$ENV{'PBCONF'}/$pbpkg/pbfilter/$dfam.pbf");
	$ffile2 = "$ENV{'PBCONF'}/$pbpkg/pbfilter/$ddir.pbf" if (-f "$ENV{'PBCONF'}/$pbpkg/pbfilter/$ddir.pbf");
	$ffile3 = "$ENV{'PBCONF'}/$pbpkg/pbfilter/$ddir-$dver.pbf" if (-f "$ENV{'PBCONF'}/$pbpkg/pbfilter/$ddir-$dver.pbf");

	push @ffiles,$ffile0 if (defined $ffile0);
	push @ffiles,$ffile1 if (defined $ffile1);
	push @ffiles,$ffile2 if (defined $ffile2);
	push @ffiles,$ffile3 if (defined $ffile3);
}
if (@ffiles) {
	print $LOG "DEBUG ffiles: ".Dumper(\@ffiles)."\n" if ($debug >= 1);

	my $config = AppConfig->new({
		# Auto Create variables mentioned in Conf file
		CREATE => 1,
		DEBUG => 0,
		GLOBAL => {
			# Each conf item is a hash
			ARGCOUNT => AppConfig::ARGCOUNT_HASH
		}
	});

	$config->file(@ffiles);
	$ptr = $config->get("filter");
	print $LOG "DEBUG f:".Dumper($ptr)."\n" if ($debug >= 1);
} else {
	$ptr = { };
}
%ptr = %$ptr;
return(\%ptr);
}

# Function which applies filter on files (only for pb)
sub pb_filter_file_pb {

my $f=shift;
my $ptr=shift;
my %filter=%$ptr;
my $destfile=shift;
my $dtype=shift;
my $dsuf=shift;
my $pbpkg=shift;
my $pbver=shift;
my $pbtag=shift;
my $pbrev=shift;
my $pbdate=shift;

print $LOG "DEBUG: From $f to $destfile\n" if ($debug >= 1);
pb_mkdir_p(dirname($destfile)) if (! -d dirname($destfile));
open(DEST,"> $destfile") || die "Unable to create $destfile";
open(FILE,"$f") || die "Unable to open $f: $!";
while (<FILE>) {
	my $line = $_;
	foreach my $s (keys %filter) {
		# Process single variables
		print $LOG "DEBUG filter{$s}: $filter{$s}\n" if ($debug >= 1);
		my $tmp = $filter{$s};
		next if (not defined $tmp);
		# Expand variables if any single one found
		print $LOG "DEBUG tmp: $tmp\n" if ($debug >= 1);
		if ($tmp =~ /\$/) {
			eval { $tmp =~ s/(\$\w+)/$1/eeg };
		# special case for ChangeLog only for pb
		} elsif (($tmp =~ /^yes$/) && ($s =~ /^PBLOG$/) && ($line =~ /^PBLOG$/)) {
			$tmp = "";
			my $p = $defpkgdir{$pbpkg};
			$p = $extpkgdir{$pbpkg} if (not defined $p);
			pb_changelog($dtype, $pbpkg, $pbtag, $dsuf, $p, \*DEST);
		}
		$line =~ s|$s|$tmp|;
	}
	print DEST $line;
}
close(FILE);
close(DEST);
}

# Function which applies filter on files (external call)
sub pb_filter_file {

my $f=shift;
my $ptr=shift;
my %filter=%$ptr;
my $destfile=shift;
my $pbpkg=shift;
my $pbver=shift;
my $pbtag=shift;
my $pbrev=shift;
my $pbdate=shift;

print $LOG "DEBUG: From $f to $destfile\n" if ($debug >= 1);
pb_mkdir_p(dirname($destfile)) if (! -d dirname($destfile));
open(DEST,"> $destfile") || die "Unable to create $destfile";
open(FILE,"$f") || die "Unable to open $f: $!";
while (<FILE>) {
	my $line = $_;
	foreach my $s (keys %filter) {
		# Process single variables
		print $LOG "DEBUG filter{$s}: $filter{$s}\n" if ($debug > 1);
		my $tmp = $filter{$s};
		next if (not defined $tmp);
		# Expand variables if any single one found
		if ($tmp =~ /\$/) {
			eval { $tmp =~ s/(\$\w+)/$1/eeg };
		}
		$line =~ s|$s|$tmp|;
	}
	print DEST $line;
}
close(FILE);
close(DEST);
}


1;
